********************************************************************************
******                        Read in the data                        **********
clear
local projectfolder "C:\Users\worri\Dropbox\Poverty\COVID19\"
cd `projectfolder'

use Monthly_Poverty_0520.dta 

gen yr_month=year*100+month   
keep if yr_month>=200501
** drop the 3/8 file in 2014 
drop if hflag==1 
** replace asecflag 2 => asecflag . 
replace asecflag=. if asecflag==2 
** drop irrelevant variables
drop month hflag qinc* qoinc* cpsid cpsidp statefip hispan durunemp whyunemp whyabsnt wnlook wkstat wksworkorg occ ind classwkr sex
** deal with missing or N/A income data
replace inctot = 0 if inctot==999999999|inctot==.


********************************************************************************
******   Construct a family unit variable for Poverty measures, famnum    ******
** For the poverty measure, a primary family and related family is considered as one family

/* Assign 1 to primary+realted families */
gen famnum = ftype==1|ftype==3 

/* Assign 2-4 to unrelated subfamilies */ 
sort yr_month asecflag serial ftype pernum
* Each subfamily has a reference person  (famrel==1)
//tab famrel if ftype==4
* Calculate # subfamilies in a household (Max 3 in this sample period)
bysort yr_month asecflag serial: egen tot_subfam = sum(ftype==4 & famrel==1)
tab tot_subfam if ftype==4 
sort yr_month asecflag serial ftype famrel
bysort yr_month asecflag serial ftype: gen subfamnum = sum(famrel==1)
tab subfamnum if ftype==4 
* Assign 2-4 to the 1st-3rd unrelated subfamily
replace famnum= subfamnum+1 if ftype==4 
drop subfamnum


/* Assign 5-20 to non-family members (ftype=2,5 or famrel=0) */
* Calculate # non-family members within a household (Max 16 in this sample period)
bysort yr_month asecflag serial: gen non_fam = sum(famrel==0) 
tab non_fam 
* Assign 5-20 to the 1st-20th non-family member. 
replace famnum= non_fam+4 if famrel==0  
drop non_fam

* Construct demographic variables related to the poverty threshold 
* 1) family size
bysort yr_month asecflag serial famnum: gen fam_size = _N
* 2) related children under age 18 (excluding HH head or spouse)
bysort yr_month asecflag serial famnum: egen nchild = sum(age<18 & relate>=301)
* 3) the head is elderly (>=65)
bysort yr_month asecflag serial famnum: egen head_elderly = max(age>=65 & relate==101)
* 4) If all members are children -> change one child to one adult (only non-householder's family)
replace nchild=nchild-1 if fam_size==nchild


* Assign poverty thresholds from the ASEC data to the relevent dem. group in the Monthly CPS
merge m:1 year fam_size nchild head_elderly using pov_thresh
keep if _merge==3
drop _merge

********************************************************************************
** Construct the family income variable in category for the ASEC sample 

* 1) identify the HH head's family  
bysort yr_month asecflag serial famnum: egen hfam = sum(relate==101) 

* 2) calculate the Householder's family income
bysort yr_month asecflag serial: egen hfaminc = sum(inctot) if hfam==1
bysort yr_month asecflag serial: egen ASECfam_inc = max(hfaminc)
replace ASECfam_inc=. if asecflag!=1
drop inctot

* 3) create an income bracket variable based on the ASEC income value
* I will impute the income based on this variable 
gen ASECincgrp = . 
replace ASECincgrp = 100 if ASECfam_inc<5000 
replace ASECincgrp = 210 if ASECfam_inc>=5000 & ASECfam_inc<7500
replace ASECincgrp = 300 if ASECfam_inc>=7500 & ASECfam_inc<10000
replace ASECincgrp = 430 if ASECfam_inc>=10000 & ASECfam_inc<12500
replace ASECincgrp = 470 if ASECfam_inc>=12500 & ASECfam_inc<15000
replace ASECincgrp = 500 if ASECfam_inc>=15000 & ASECfam_inc<20000
replace ASECincgrp = 600 if ASECfam_inc>=20000 & ASECfam_inc<25000
replace ASECincgrp = 710 if ASECfam_inc>=25000 & ASECfam_inc<30000 
replace ASECincgrp = 720 if ASECfam_inc>=30000 & ASECfam_inc<35000
replace ASECincgrp = 730 if ASECfam_inc>=35000 & ASECfam_inc<40000 
replace ASECincgrp = 740 if ASECfam_inc>=40000 & ASECfam_inc<50000
replace ASECincgrp = 820 if ASECfam_inc>=50000 & ASECfam_inc<60000
replace ASECincgrp = 830 if ASECfam_inc>=60000 & ASECfam_inc<75000
replace ASECincgrp = 840 if ASECfam_inc>=75000 & year<=2004
replace ASECincgrp = 841 if ASECfam_inc>=75000 & ASECfam_inc<100000  & year>=2005
replace ASECincgrp = 842 if ASECfam_inc>=100000 & ASECfam_inc<150000 & year>=2005
replace ASECincgrp = 843 if ASECfam_inc>=150000 & year>=2005
* Assign the original family income for the Monthly CPS sample
replace ASECincgrp = faminc if asecflag!=1
tab ASECincgrp if year==2020



** construct 15 demographic groups based on householder's family characteristics
gen gr=.
replace gr=1  if fam_size==1 & head_elderly==0
replace gr=2  if fam_size==1 & head_elderly==1
replace gr=3  if fam_size==2 & head_elderly==0 & nchild==0 
replace gr=4  if fam_size==2 & head_elderly==1 & nchild==0 
replace gr=5  if fam_size==2 & nchild==1
replace gr=6  if fam_size==3 & nchild==0
replace gr=7  if fam_size==3 & nchild==1
replace gr=8  if fam_size==3 & nchild==2
replace gr=9  if fam_size==4 & nchild<=1
replace gr=10 if fam_size==4 & nchild==2
replace gr=11 if fam_size==4 & nchild==3
replace gr=12 if fam_size==5 & nchild<=2
replace gr=13 if fam_size==5 & nchild>=3
replace gr=14 if fam_size==6 
replace gr=15 if fam_size>=7


********************************************************************************
** Family Income Imputation within income-demographic cell using the ASEC Income
*1) convert the data to the household level data  
preserve 
keep year yr_month asecflag serial relate ASECincgrp gr faminc ASECfam_inc 
* keep one obs per HH (head)
sort yr_month asecflag serial relate
bysort yr_month asecflag serial: keep if _n==1
* calculate the size of demographic cells in each ASEC income bracket 
bysort year ASECincgrp gr asecflag: gen ASEC_N = _N if asecflag==1 
bysort year ASECincgrp gr: egen N = max(ASEC_N) 
* the 2020 ASEC data is not available yet. Assign the 2019 ASEC number
bysort year ASECincgrp gr asecflag: gen ASEC_N19 = _N if asecflag==1 & year==2019
bysort ASECincgrp gr: egen ASECN19 = max(ASEC_N19) 
replace N= ASECN19 if year==2020
drop ASEC_N ASEC_N19 ASECN19 

*2) within a cell, assign to monthly CPS HH a random number btw 1 and N, where N is the size of each cell in the ASEC sample
* this random number will be used when assigning ASEC income to monthly CPS households 
sort yr_month asecflag faminc gr serial
set seed 1000126
bysort yr_month asecflag faminc gr: gen rnum1 = floor(N*runiform()+1)
tempfile hh_level
save `hh_level', replace

*3) create a new ASEC sample with random ASEC income within each cell 
keep if asecflag==1 
set seed 1000126
bsample, strata(year ASECincgrp gr) 
* the 2020 ASEC data is not available yet. Assign the 2019 ASEC number
expand 2 if year==2019, gen(ASEC20)
replace year=2020 if ASEC20==1
drop rnum1 ASEC20
sort year ASECincgrp gr yr_month serial
bysort year ASECincgrp gr: gen rnum1 = _n 
rename ASECfam_inc ifam_inc
keep year ASECincgrp gr rnum1 ifam_inc 
tempfile bsample
save `bsample'
restore 

*4) within a cell, assign the random draw of ASEC income to monthly CPS households 
preserve
use `hh_level', clear
tab year
merge m:1 year ASECincgrp gr rnum1 using `bsample'
* check whether there is missing cells in the ASEC data
tab faminc gr if _merge==1 & faminc<844
tab yr_month if _merge==1 & faminc<844
drop if _merge==2
keep yr_month asecflag serial ifam_inc
save impute, replace
restore

*5) merge this household level data to the original individual level data
merge m:1 yr_month asecflag serial using impute
drop _merge
* the imputed income (ifam_inc should fall into faminc for the monthly CPS sample 
bysort year asecflag faminc: sum ifam_inc   




*******************         End of the data cleaning           *****************
********************************************************************************
*** Employment status and earnings among workers 
* Scale variables
gen scale=((fam_size-nchild+.7*nchild)^.7)

* Employment status
gen emp = empstat==10|empstat==12
drop empstat  
 

********************************************************************************
* Employment Status (Appendix Figure 1)
preserve
drop if asecflag==1 
keep yr_month year age emp wtfinl 
keep if age>=16      //match BLS samples
collapse (mean) year emp [w=wtfinl], by(yr_month)
gen mo = mod(yr_month, 100)
export excel using App.F1.xlsx, keepcellfmt firstrow(var) sheet(full) replace
restore



* Earnings (Appendix Figure 2)
preserve
** Keep householder's family only 
keep if hfam==1
drop if asecflag==1 
keep yr_month asecflag serial famnum scale age mish earnweek earnwt
merge m:1 yr_month using PCE
drop _merge
replace earnweek=0 if earnweek>9999
bysort yr_month asecflag serial famnum: egen totearn = sum(PCE*2.355*earnweek*4.3/scale)
keep if age>=16      //match BLS samples
gen earning = totearn if (mish==4|mish==8)
collapse earning [w=earnwt], by(yr_month)
gen mo = mod(yr_month, 100)
export excel using App.F2.xlsx, keepcellfmt firstrow(var) replace
restore



********************************************************************************
*** Restrict the sample to householder's family 
keep if hfam==1
* Missing incomes or CPS-imputed income 
gen miss_inc = (faminc>=995 & faminc<=999)|(qfaminc>0 & qfaminc!=.)


********************************************************************************
**  Construct Fixed Demographic Weight (Using the demographic composition in Jan & Feb 2020)
merge 1:1 yr_month asecflag serial pernum using fixedwgt.dta
** for March 2020 onward, use the fixed demographic weight
replace wtfinl = fixedwgt if yr_month>=202003


********************************************************************************
* Employment Status (Appendix Figure 1)
preserve
drop if asecflag==1 
keep yr_month year age emp mish wtfinl miss_inc 
keep if age>=16      //match BLS samples
foreach var in emp {
gen mo15_e = `var' if miss_inc!=1 & (mish==1|mish==5) 
gen mo5_e = `var' if miss_inc!=1 & (mish==5) 
}
collapse (mean) year mo15_e mo5_e [w=wtfinl], by(yr_month)
gen mo = mod(yr_month, 100)
export excel using App.F1.xlsx, keepcellfmt firstrow(var) sheet(1st5th) 
restore




********************************************************************************
***********            Figures 1 and 3: Poverty Rates            *************** 
********************************************************************************

forvalues n=1/1 {
gen pov`n'= ifam_inc<pov_thresh
}
replace pov1=. if miss_inc==1|(mish!=1 & mish!=5) //1st and 5th month only
//replace pov2=. if miss_inc==1|(mish!=5)  // 5th month only

** pov_ASEC: ASEC poverty
gen pov_ASEC = ASECfam_inc<pov_thresh  
** replicate the official poverty rate (using the analysis sample)
//gen pov_off= offtotval<offcutoff 


* Monthly CPS Poverty rate
preserve
keep asecflag year pov1 wtfinl yr_month
keep if asecflag!=1 
collapse (mean) year pov1 [w=wtfinl], by(yr_month)
tempfile mo_CPS_pov
save `mo_CPS_pov'
restore


* CPS ASEC Poverty rate
preserve
keep asecflag year pov_ASEC asecwt yr_month
keep if asecflag==1 
collapse (mean) pov_ASEC [w=asecwt], by(year)
replace year=year-1 //reference year
merge 1:m year using `mo_CPS_pov'
drop _merge year
sort yr_month 
** Monthly CPS Poverty rate 3mo MA poverty rates 
gen pov1_3mo = (pov1[_n-1]+pov1[_n]+pov1[_n+1])/3
  
** Official poverty rate from the U.S. Census report  
gen pov_off = .
replace pov_off = 0.126 if yr_month==200512
replace pov_off = 0.123 if yr_month==200612
replace pov_off = 0.125 if yr_month==200712
replace pov_off = 0.132 if yr_month==200812
replace pov_off = 0.143 if yr_month==200912
replace pov_off = 0.151 if yr_month==201012
replace pov_off = 0.150 if yr_month==201112
replace pov_off = 0.150 if yr_month==201212
replace pov_off = 0.145 if yr_month==201312
replace pov_off = 0.148 if yr_month==201412
replace pov_off = 0.135 if yr_month==201512
replace pov_off = 0.127 if yr_month==201612
replace pov_off = 0.123 if yr_month==201712
replace pov_off = 0.118 if yr_month==201812
order yr_month pov1 pov1_3mo pov_ASEC pov_off 
gen mo = mod(yr_month, 100)
replace pov_ASEC=. if mo!=12
drop mo
export excel using Figure1&3.xlsx, keepcellfmt firstrow(var) replace
restore

********************************************************************************
******   Figures 2 and 4: 10, 25, 50, and 75th income percentiles       ******** 
********************************************************************************

* Monthly CPS income percentile (1st and 5th month)
preserve
keep asecflag ifam_inc scale miss_inc mish year wtfinl yr_month 
keep if asecflag!=1  
merge m:1 yr_month using PCE
drop _merge
replace ifam_inc=. if miss_inc==1|(mish!=1 & mish!=5)   // 1st and 5th month
replace ifam_inc = PCE*2.355*ifam_inc/scale   // equivalent scale adjustment & 2adults and 2 children
collapse (mean) year (p75) p75 = ifam_inc (p50) p50 = ifam_inc (p25) p25 = ifam_inc (p10) p10 = ifam_inc  [w=wtfinl], by(yr_month)
tempfile mo_CPS_pct
save `mo_CPS_pct'
restore


* ASEC income percentile 
preserve
keep asecflag ASECfam_inc scale year asecwt yr_month 
keep if asecflag==1  
merge m:1 yr_month using PCE
drop _merge
replace ASECfam_inc = PCE*2.355*ASECfam_inc/scale   // equivalent scale adjustment & 2adults and 2 children
collapse (p10) ASECp10 = ASECfam_inc (p25) ASECp25 = ASECfam_inc (p50) ASECp50 = ASECfam_inc (p75) ASECp75 = ASECfam_inc [w=asecwt], by(year)
replace year=year-1 //reference year
merge 1:m year using `mo_CPS_pct'
drop _merge year
sort yr_month 
gen mo = mod(yr_month, 100)
order yr_month p75 p50 p25 p10 ASEC*
foreach pct in 10 25 50 75 {
replace ASECp`pct'=. if mo!=12
}
drop mo
export excel using Figure2&4.xlsx, keepcellfmt firstrow(var) replace
restore


********************************************************************************
***  Figure 5: Quarterly Income Poverty rate from the CPS and CE Data    *******


* Monthly CPS
preserve
keep if miss_inc!=1 & (mish==1|mish==5) 
keep if yr_month>=201401 & yr_month<=201812
keep hfam ftype asecflag year pov1 pov2 wtfinl yr_month
gen mo = mod(yr_month, 100)
gen q = mo>=1 & mo<=3
replace q = 2 if mo>=4 & mo<=6
replace q = 3 if mo>=7 & mo<=9
replace q = 4 if mo>=10 & mo<=12
gen yrq = year*10+q
keep if asecflag!=1 
collapse (mean) pov1 [w=wtfinl], by(yrq)
tempfile qpov_CPS
save `qpov_CPS'
restore


* Consumer Expenditure Survey 
preserve
clear
use CEdata_for_BPEA.dta
keep if ref_year>=2013
rename ref_year year
rename perslt18 nchild
gen head_elderly=age_ref>=65
* topcode family size at 16
replace fam_size=16 if fam_size>=16
* make sure that at least one person is an adult
replace nchild=nchild-1 if fam_size==nchild

order year fam_size nchild head_elderly
merge m:1 year fam_size nchild head_elderly using pov_thresh
keep if _merge==3
drop _merge


* Money Income Poverty
gen inc = fincbtax-fstamp
gen cepov = inc<pov_thresh

* treat the previous year's 5th quarter data as the current year 1st quarter data  
replace qyear = 1051 if qyear ==1045 
replace qyear = 1151 if qyear ==1145 
replace qyear = 1191 if qyear ==1185 
keep if qyear>=1051 & qyear!=.

gen yrq = 19000+qyear
collapse (mean) cepov [aw = wgt20], by(yrq)
merge 1:1 yrq using `qpov_CPS'
keep if _merge==3
drop _merge 
order yrq pov1 cepov 
export excel using Figure5.xlsx, keepcellfmt firstrow(var) replace
restore


********************************************************************************
********          plot figures using the data in excel files         ***********
********************************************************************************
*** Figure 1
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\Figure1&3.xlsx", sheet("Sheet1") firstrow clear
keep if yr_month>=201901 & yr_month<=202006
gen date = _n
gen yrm = tm(2019m1) + date -1
sum yrm 
format yrm %tm
format yrm %tmMon-YY
label variable pov1 "Monthly CPS Poverty"
label variable pov1_3mo "Monthly CPS Poverty, 3-month moving average"
twoway (scatter pov1 pov1_3mo yrm, connect(l l) lwidth(thick thick) lcolor(red blue) lp(solid dash) msymbol(i i) mcolor(i i) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(2) region(lcolor(ltblue)) size(small))), ytitle(Fraction Poor, size(medsmall)) xtitle("") ///
title("Figure 1: Poverty Rates from the Monthly CPS, 2019-2020", size(medsmall)) ///
ylabel(0.00(0.02)0.14, angle(horizontal)) xlabel(708(1)725, angle(vertical))
graph export "Figure1.png", replace 


********************************************************************************
*** Figure 2
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\Figure2&4.xlsx", sheet("Sheet1") firstrow clear
keep if yr_month>=201901 & yr_month<=202006
gen date = _n
gen yrm = tm(2019m1) + date -1
sum yrm 
format yrm %tm
format yrm %tmMon-YY
label variable p75 "Monthly CPS, 75th percentile"
label variable p50 "Monthly CPS, 50th percentile"
label variable p25 "Monthly CPS, 25th percentile"
label variable p10 "Monthly CPS, 10th percentile"

twoway (line p75 p50 p25 p10 yrm, lwidth(thick thick thick thick) lcolor(orange gray sandb eltblue) lp(solid shortdash longdash dash) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(2) region(lcolor(ltblue)) size(small))), ytitle(2020$, size(medsmall)) xtitle("") ///
title("Figure 2: Percentiles of Family Income from the Monthly CPS, 2019-2020", size(medsmall)) ///
ylabel(0.00(20000)160000, angle(horizontal) format(%9.0gc)) xlabel(708(1)725, angle(vertical))
graph export "Figure2.png", replace 


********************************************************************************
*** Figure 3
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\Figure1&3.xlsx", sheet("Sheet1") firstrow clear
keep if yr_month>=200501 & yr_month<=202006
gen date = _n
gen yrm = tm(2005m1) + date -1
sum yrm 
format yrm %tm
format yrm %tmMon-YY
label variable pov1 "Monthly CPS Poverty"
label variable pov1_3mo "Monthly CPS Poverty, 3-month moving average"
label variable pov_ASEC "CPS ASEC Poverty"
label variable pov_off "Official Poverty (All families and individuals)"

twoway (scatter pov1 pov1_3mo pov_ASEC pov_off yrm, connect(l l) lwidth(thick thick) lcolor(eltblue blue) msymbol(i i s t) mcolor(i i sandb orange) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(2) region(lcolor(ltblue)) size(small))), ytitle(Fraction Poor, size(medsmall)) xtitle("") ///
title("Figure 3: Poverty Rates from the Monthly CPS and" "the Annual Social and Economic Survey of the CPS, 2005-2020", size(medsmall)) ///
ylabel(0.06(0.02)0.18, angle(horizontal)) xlabel(545(6)725, angle(vertical)) 
graph export "Figure3.png", replace 


********************************************************************************
*** Figure 4
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\Figure2&4.xlsx", sheet("Sheet1") firstrow clear
keep if yr_month>=200501 & yr_month<=202006
gen date = _n
gen yrm = tm(2005m1) + date -1
sum yrm 
format yrm %tm
format yrm %tmMon-YY
label variable p75 "Monthly CPS, 75th percentile"
label variable p50 "Monthly CPS, 50th percentile"
label variable p25 "Monthly CPS, 25th percentile"
label variable p10 "Monthly CPS, 10th percentile"
label variable ASECp75 "CPS ASEC, 75th percentile"
label variable ASECp50 "CPS ASEC, 50th percentile"
label variable ASECp25 "CPS ASEC, 25th percentile"
label variable ASECp10 "CPS ASEC, 10th percentile"

twoway (scatter p75 p50 p25 p10 ASECp75 ASECp50 ASECp25 ASECp10 yrm, connect(l l l l) lwidth(thick thick thick thick) lcolor(orange gray sandb eltblue) msymbol(i i i i s s s s) mcolor(i i i i orange gray sandb eltblue) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(4) region(lcolor(ltblue)) size(small))), ytitle(2020$, size(medsmall)) xtitle("") ///
title("Figure 4: Percentiles of Family Income from the Monthly CPS and the Annual Social" "and Economic Survey of the CPS, 2005-2020", size(small)) ///
ylabel(0.00(20000)160000, angle(horizontal) format(%9.0gc)) xlabel(, angle(vertical)) xlabel(545(6)725) 
graph export "Figure4.png", replace 

********************************************************************************
*** Figure 5
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\Figure5.xlsx", sheet("Sheet1") firstrow clear
gen date = _n
gen q = tq(2014q1) + date -1
sum q 
format q %tq
format q %tqq-YY
label variable pov1 "Monthly CPS Poverty"
label variable cepov "CE Income Poverty"
twoway (line pov1 cepov q, lwidth(thick thick) lcolor(red blue) lp(solid dash) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(2) region(lcolor(ltblue)) size(small))), ytitle(Fraction Poor, size(medsmall)) xtitle("") ///
title("Figure 5: Poverty Rates from the Monthly CPS and" "the The Consumer Expenditure Surveys, 2014-2018", size(medsmall)) ///
ylabel(0.06(0.02)0.18, angle(horizontal)) xlabel(216(2)235) 
graph export "Figure5.png", replace 


********************************************************************************
*** Appendix Figure 1
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\App.F1.xlsx", sheet("full") firstrow clear
preserve
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\App.F1.xlsx", sheet("1st5th") firstrow clear
keep yr_month mo15_e
tempfile emp
save `emp'
restore
merge 1:1 yr_month using `emp'
keep if yr_month>=201901 & yr_month<=202006
gen date = _n
gen yrm = tm(2019m1) + date -1
sum yrm 
format yrm %tmMon-YY
label variable emp  "Full Sample"
label variable mo15_e "Analysis Sample"
twoway (line emp mo15_e yrm, lwidth(thick thick) lcolor(red blue) lp(solid dash) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(2) region(lcolor(ltblue)) size(small))), ytitle(Fraction Employed, size(medsmall)) xtitle("") ///
title("Appendix Figure 1: Monthly Employment Status, Monthly CPS, 2019-2020", size(medsmall)) ///
ylabel(0.40(0.05)0.65, angle(horizontal)) xlabel(708(1)725, angle(vertical))
graph export "AppF1.png", replace 

********************************************************************************
*** Appendix Figure 2
import excel "C:\Users\worri\Dropbox\Poverty\COVID19\App.F2.xlsx", sheet("Sheet1") firstrow clear
keep if yr_month>=201901 & yr_month<=202006
gen date = _n
gen yrm = tm(2019m1) + date -1
sum yrm 
format yrm %tmMon-YY
label variable earning  "Full Sample"
twoway (line earning yrm, lwidth(thick) lcolor(blue) ///
graphregion(fcolor(white)) bgcolor(white) legend(rows(1) region(lcolor(ltblue)) size(small))), ytitle(2020$, size(medsmall)) xtitle("") ///
title("Appendix Figure 2: Mean Monthly Family Earnings, Monthly CPS, 2019-2020", size(medsmall)) ///
ylabel(4000(500)7000, angle(horizontal) format(%9.0gc)) xlabel(708(1)725, angle(vertical))
graph export "AppF2.png", replace 


********************************************************************************
*********       Regression of ASEC poverty on Monthly poverty        ***********
/*
clear all
import excel "`projectfolder'Figure1&3.xlsx", sheet("Sheet1") firstrow
drop _merge
preserve 
regress pov1 pov_ASEC
regress pov_ASEC pov1 
corr pov1 pov_ASEC
sum pov_ASEC pov1, detail
**********       Regression of CE poverty on Monthly poverty        ************
clear
import excel "`projectfolder'Figure5.xlsx", sheet("Sheet1") firstrow
gen q = mod(yrq, 10)
regress pov1 cepov
regress cepov pov1 
corr pov1 cepov
sum cepov pov1, detail


